<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Category;
use Illuminate\Support\Str;

class CategoryController extends Controller
{
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:categories,name',
        ]);

        $category = Category::create([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'description' => '',
        ]);

        return response()->json([
            'success' => true,
            'category' => $category,
        ]);
    }

    // Hapus semua konten dalam kategori tertentu
    public function deleteContents(Request $request, Category $category)
    {
        // Ambil semua konten dalam kategori
        $contents = $category->contents;
        // Ambil menu_id dari salah satu konten (jika ada)
        $menuId = $contents->first() ? $contents->first()->menu_id : null;
        // Hapus semua konten
        foreach ($contents as $content) {
            // Hapus file gambar jika ada
            if ($content->image) {
                \Storage::delete('public/' . $content->image);
            }
            $content->delete();
        }
        // Redirect ke halaman menu.show jika menuId ada, jika tidak ke halaman sebelumnya
        if ($menuId) {
            return redirect()->route('menus.show', $menuId)->with('success', 'Semua konten dalam kategori berhasil dihapus!');
        } else {
            return redirect()->back()->with('success', 'Semua konten dalam kategori berhasil dihapus!');
        }
    }

    // ========== API METHODS ==========

    /**
     * API: Get all categories
     */
    public function apiIndex()
    {
        $categories = Category::with('contents')->get();
        return response()->json([
            'success' => true,
            'data' => $categories
        ]);
    }

    /**
     * API: Store new category
     */
    public function apiStore(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:categories,name',
            'description' => 'nullable|string'
        ]);

        $category = Category::create([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'description' => $request->description ?? '',
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Category created successfully',
            'data' => $category
        ], 201);
    }

    /**
     * API: Get specific category
     */
    public function apiShow(Category $category)
    {
        $category->load('contents');
        return response()->json([
            'success' => true,
            'data' => $category
        ]);
    }

    /**
     * API: Update category
     */
    public function apiUpdate(Request $request, Category $category)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:categories,name,' . $category->id,
            'description' => 'nullable|string'
        ]);

        $category->update([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'description' => $request->description ?? $category->description,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Category updated successfully',
            'data' => $category
        ]);
    }

    /**
     * API: Delete category
     */
    public function apiDestroy(Category $category)
    {
        // Check if category has contents
        if ($category->contents()->count() > 0) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete category that has contents'
            ], 400);
        }

        $category->delete();

        return response()->json([
            'success' => true,
            'message' => 'Category deleted successfully'
        ]);
    }
} 