<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Content;

class ContentController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $contents = Content::latest()->get(); // tampilkan semua data tanpa pagination
        return view('content.index', compact('contents'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $menus = \App\Models\Menu::where('status', 1)->get();
        $categories = \App\Models\Category::all();
        $selectedMenuId = $request->get('menu_id');
        return view('content.create', compact('menus', 'categories', 'selectedMenuId'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'content' => 'required|string',
            'menu_id' => 'required|exists:menus,id',
            'category_id' => 'nullable|exists:categories,id',
            'status' => 'required|in:draft,published',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        $data = $request->all();
        
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '.' . $image->getClientOriginalExtension();
            $image->storeAs('public/content', $imageName);
            $data['image'] = 'content/' . $imageName;
            // Tambahkan copy manual ke public/storage jika symlink tidak ada
            $from = storage_path('app/public/content/' . $imageName);
            $to = public_path('storage/content/' . $imageName);
            if (file_exists($from)) {
                @mkdir(dirname($to), 0755, true);
                copy($from, $to);
            }
        }

        $data['user_id'] = auth()->id();
        
        $content = Content::create($data);

        return redirect()->route('menus.show', $content->menu_id)
            ->with('success', 'Konten berhasil dibuat!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Content $content)
    {
        return view('content.show', compact('content'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Content $content)
    {
        $menus = \App\Models\Menu::where('status', 1)->get();
        $categories = \App\Models\Category::all();
        return view('content.edit', compact('content', 'menus', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Content $content)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'content' => 'required|string',
            'menu_id' => 'required|exists:menus,id',
            'category_id' => 'nullable|exists:categories,id',
            'status' => 'required|in:draft,published',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        $data = $request->all();
        
        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($content->image) {
                \Storage::delete('public/' . $content->image);
            }
            
            $image = $request->file('image');
            $imageName = time() . '.' . $image->getClientOriginalExtension();
            $image->storeAs('public/content', $imageName);
            $data['image'] = 'content/' . $imageName;
            // Tambahkan copy manual ke public/storage jika symlink tidak ada
            $from = storage_path('app/public/content/' . $imageName);
            $to = public_path('storage/content/' . $imageName);
            if (file_exists($from)) {
                @mkdir(dirname($to), 0755, true);
                copy($from, $to);
            }
        }

        $content->update($data);

        return redirect()->route('menus.show', $content->menu_id)
            ->with('success', 'Konten berhasil diperbarui!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Content $content)
    {
        // Delete image if exists
        if ($content->image) {
            \Storage::delete('public/' . $content->image);
        }
        
        $menuId = $content->menu_id;
        $content->delete();

        return redirect()->route('menus.show', $menuId)
            ->with('success', 'Konten berhasil dihapus!');
    }

    // ========== API METHODS ==========

    /**
     * API: Get all contents
     */
    public function apiIndex()
    {
        $host = 'https://tisel.pthas.or.id';
        $contents = Content::with(['menu', 'category', 'user'])->latest()->get()->map(function($content) use ($host) {
            $arr = $content->toArray();
            if ($content->image) {
                $arr['image'] = $host . '/storage/' . ltrim($content->image, '/');
            }
            return $arr;
        });
        return response()->json([
            'success' => true,
            'data' => $contents
        ]);
    }

    /**
     * API: Store new content
     */
    public function apiStore(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'content' => 'required|string',
            'menu_id' => 'required|exists:menus,id',
            'category_id' => 'nullable|exists:categories,id',
            'status' => 'required|in:draft,published',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        $data = $request->all();
        $data['user_id'] = auth()->id();
        
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '.' . $image->getClientOriginalExtension();
            $image->storeAs('public/content', $imageName);
            $data['image'] = 'content/' . $imageName;
            // Tambahkan copy manual ke public/storage jika symlink tidak ada
            $from = storage_path('app/public/content/' . $imageName);
            $to = public_path('storage/content/' . $imageName);
            if (file_exists($from)) {
                @mkdir(dirname($to), 0755, true);
                copy($from, $to);
            }
        }
        
        $content = Content::create($data);
        $content->load(['menu', 'category', 'user']);

        return response()->json([
            'success' => true,
            'message' => 'Content created successfully',
            'data' => $content
        ], 201);
    }

    /**
     * API: Get specific content
     */
    public function apiShow(Content $content)
    {
        $host = 'https://tisel.pthas.or.id';
        $content->load(['menu', 'category', 'user']);
        $arr = $content->toArray();
        if ($content->image) {
            $arr['image'] = $host . '/storage/' . ltrim($content->image, '/');
        }
        return response()->json([
            'success' => true,
            'data' => $arr
        ]);
    }

    /**
     * API: Update content
     */
    public function apiUpdate(Request $request, Content $content)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'content' => 'required|string',
            'menu_id' => 'required|exists:menus,id',
            'category_id' => 'nullable|exists:categories,id',
            'status' => 'required|in:draft,published',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        $data = $request->all();
        
        if ($request->hasFile('image')) {
            if ($content->image) {
                \Storage::delete('public/' . $content->image);
            }
            
            $image = $request->file('image');
            $imageName = time() . '.' . $image->getClientOriginalExtension();
            $image->storeAs('public/content', $imageName);
            $data['image'] = 'content/' . $imageName;
            // Tambahkan copy manual ke public/storage jika symlink tidak ada
            $from = storage_path('app/public/content/' . $imageName);
            $to = public_path('storage/content/' . $imageName);
            if (file_exists($from)) {
                @mkdir(dirname($to), 0755, true);
                copy($from, $to);
            }
        }

        $content->update($data);
        $content->load(['menu', 'category', 'user']);

        return response()->json([
            'success' => true,
            'message' => 'Content updated successfully',
            'data' => $content
        ]);
    }

    /**
     * API: Delete content
     */
    public function apiDestroy(Content $content)
    {
        if ($content->image) {
            \Storage::delete('public/' . $content->image);
        }
        
        $content->delete();

        return response()->json([
            'success' => true,
            'message' => 'Content deleted successfully'
        ]);
    }
} 