<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Contoh;
use App\Models\Content;

class ContohController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $content_id = $request->input('content_id');
        return view('contoh.create', compact('content_id'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'nama' => 'required|string|max:255',
            'fokus_utama' => 'required|string|max:255',
            'aspek_unik' => 'required|string|max:255',
            'isi' => 'required|string',
        ]);
        $contoh = Contoh::create($request->only(['nama', 'fokus_utama', 'aspek_unik', 'isi', 'content_id']));
        // Ambil menu_id dari content terkait
        $content = $contoh->content;
        $menuId = $content ? $content->menu_id : null;
        if ($menuId) {
            return redirect()->route('menus.show', $menuId)
                ->with('success', 'Contoh berhasil ditambahkan!');
        } else {
            return redirect()->back()->with('success', 'Contoh berhasil ditambahkan!');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Contoh $contoh)
    {
        return view('contoh.show', compact('contoh'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Contoh $contoh)
    {
        return view('contoh.edit', compact('contoh'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $request->validate([
            'nama' => 'required|string|max:255',
            'fokus_utama' => 'required|string|max:255',
            'aspek_unik' => 'required|string|max:255',
            'isi' => 'required|string',
        ]);
        $contoh = Contoh::findOrFail($id);
        $contoh->update($request->only(['nama', 'fokus_utama', 'aspek_unik', 'isi']));
        $content = $contoh->content;
        $menuId = $content ? $content->menu_id : null;
        if ($menuId) {
            return redirect()->route('menus.show', $menuId)
                ->with('success', 'Contoh berhasil diupdate!');
        } else {
            return redirect()->back()->with('success', 'Contoh berhasil diupdate!');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $contoh = Contoh::findOrFail($id);
        $contoh->delete();
        return redirect()->back()->with('success', 'Contoh berhasil dihapus!');
    }

    // ========== API METHODS ==========

    /**
     * API: Get all contoh
     */
    public function apiIndex()
    {
        $contoh = Contoh::with('content')->latest()->get();
        return response()->json([
            'success' => true,
            'data' => $contoh
        ]);
    }

    /**
     * API: Store new contoh
     */
    public function apiStore(Request $request)
    {
        $request->validate([
            'nama' => 'required|string|max:255',
            'fokus_utama' => 'required|string|max:255',
            'aspek_unik' => 'required|string|max:255',
            'isi' => 'required|string',
            'content_id' => 'required|exists:contents,id'
        ]);

        $contoh = Contoh::create($request->only(['nama', 'fokus_utama', 'aspek_unik', 'isi', 'content_id']));
        $contoh->load('content');

        return response()->json([
            'success' => true,
            'message' => 'Contoh created successfully',
            'data' => $contoh
        ], 201);
    }

    /**
     * API: Get specific contoh
     */
    public function apiShow(Contoh $contoh)
    {
        $contoh->load('content');
        return response()->json([
            'success' => true,
            'data' => $contoh
        ]);
    }

    /**
     * API: Update contoh
     */
    public function apiUpdate(Request $request, Contoh $contoh)
    {
        $request->validate([
            'nama' => 'required|string|max:255',
            'fokus_utama' => 'required|string|max:255',
            'aspek_unik' => 'required|string|max:255',
            'isi' => 'required|string',
        ]);

        $contoh->update($request->only(['nama', 'fokus_utama', 'aspek_unik', 'isi']));
        $contoh->load('content');

        return response()->json([
            'success' => true,
            'message' => 'Contoh updated successfully',
            'data' => $contoh
        ]);
    }

    /**
     * API: Delete contoh
     */
    public function apiDestroy(Contoh $contoh)
    {
        $contoh->delete();

        return response()->json([
            'success' => true,
            'message' => 'Contoh deleted successfully'
        ]);
    }
}
