<?php

namespace App\Http\Controllers;

use App\Models\Refleksi;
use App\Models\Content;
use Illuminate\Http\Request;

class RefleksiController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function create(Request $request)
    {
        $content_id = $request->get('content_id');
        $content = Content::findOrFail($content_id);
        return view('refleksi.create', compact('content_id', 'content'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'content_id' => 'required|exists:contents,id',
            'refleksi' => 'required|string',
        ]);
        $data = $request->only(['content_id', 'refleksi']);
        $data['user_id'] = auth()->id();
        $refleksi = Refleksi::create($data);

        // Redirect ke halaman asal jika ada, jika tidak ke root
        return redirect($request->redirect_to ?? '/')->with('success', 'Refleksi berhasil ditambahkan!');
    }

    public function show(Refleksi $refleksi)
    {
        return view('refleksi.show', compact('refleksi'));
    }

    public function edit(Refleksi $refleksi)
    {
        $this->authorize('update', $refleksi);
        return view('refleksi.edit', compact('refleksi'));
    }

    public function update(Request $request, Refleksi $refleksi)
    {
        $this->authorize('update', $refleksi);
        $request->validate([
            'refleksi' => 'required|string',
        ]);
        $refleksi->update($request->only(['refleksi']));
        // Ambil menu_id dari content terkait
        $content = \App\Models\Content::find($refleksi->content_id);
        $menuId = $content ? $content->menu_id : null;
        if ($menuId) {
            return redirect()->route('menus.show', $menuId)
                ->with('success', 'Refleksi berhasil diupdate!');
        } else {
            return redirect('/')->with('success', 'Refleksi berhasil diupdate!');
        }
    }

    public function destroy(Refleksi $refleksi)
    {
        $this->authorize('delete', $refleksi);
        $content_id = $refleksi->content_id;
        $refleksi->delete();
        return redirect()->route('content.show', $content_id)
            ->with('success', 'Refleksi berhasil dihapus!');
    }

    // ========== API METHODS ==========

    /**
     * API: Get all refleksi
     */
    public function apiIndex()
    {
        $refleksi = Refleksi::with(['content', 'user'])->latest()->get();
        return response()->json([
            'success' => true,
            'data' => $refleksi
        ]);
    }

    /**
     * API: Store new refleksi
     */
    public function apiStore(Request $request)
    {
        $request->validate([
            'content_id' => 'required|exists:contents,id',
            'refleksi' => 'required|string',
        ]);

        $data = $request->only(['content_id', 'refleksi']);
        $data['user_id'] = auth()->id();
        
        $refleksi = Refleksi::create($data);
        $refleksi->load(['content', 'user']);

        return response()->json([
            'success' => true,
            'message' => 'Refleksi created successfully',
            'data' => $refleksi
        ], 201);
    }

    /**
     * API: Get specific refleksi
     */
    public function apiShow(Refleksi $refleksi)
    {
        $refleksi->load(['content', 'user']);
        return response()->json([
            'success' => true,
            'data' => $refleksi
        ]);
    }

    /**
     * API: Update refleksi
     */
    public function apiUpdate(Request $request, Refleksi $refleksi)
    {
        // Check if user can update this refleksi
        if ($refleksi->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized to update this refleksi'
            ], 403);
        }

        $request->validate([
            'refleksi' => 'required|string',
        ]);

        $refleksi->update($request->only(['refleksi']));
        $refleksi->load(['content', 'user']);

        return response()->json([
            'success' => true,
            'message' => 'Refleksi updated successfully',
            'data' => $refleksi
        ]);
    }

    /**
     * API: Delete refleksi
     */
    public function apiDestroy(Refleksi $refleksi)
    {
        // Check if user can delete this refleksi
        if ($refleksi->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized to delete this refleksi'
            ], 403);
        }

        $refleksi->delete();

        return response()->json([
            'success' => true,
            'message' => 'Refleksi deleted successfully'
        ]);
    }
} 